/** @file   main.cpp
 * @brief   Main function definition
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */
 
/**
 * @mainpage
 * @author    Tomi Lamminsaari
 * 
 * @par Idea of the game in brief
 * We Want War is a classic top-down shooting game where one brave
 * storeman saves the world from alien invasion. The goal is to
 * make addictive shoot'em up game with smooth gameplay and lot of
 * action.
 *
 * @par Development enviroment
 * To compile the We Want War you need to have two external libraries
 * installed. One is Allegro and the other one is AllegroMP3.
 * @par
 * This game has been developped with Dev-C++ IDE and MinGW32 compiler.
 * Documentation is created with Doxygen and in revision control I've
 * been using the Component Software RCS.
 *
 * @par Brief glance to the structure of We Want War
 * The <code> Game </code> - class is where everything begins. The main-function
 * creates an istance of this class and it is responssible of initializing
 * the graphics, sounds, animations, etc. It also manages the menupart
 * of the game.
 * @par
 * Each menupage is an instance of its own class. All these menupage classes
 * are derived from <code> MenuPage </code> - class. These menupage classes
 * then implement  both the <code> update() </code> and
 * <code> redraw() </code> - methods allowing them to behave and look different.
 *
 * @par
 * When the shooting begins, the <code> War </code> - class steps in. The
 * <code> Game </code> object creates a new instance of <code> War </code> -
 * class for every game level. It takes care of the actionpart of the game.
 * At first it loads the level Map, creates the <code>Alien</code>
 * and <code>Player</code> objects as well as all the other objects.
 * <code>War</code> - class takes care of redrawing most of the graphics.
 * @par
 * Alien and Player objects each have their own classes. They all are derived
 * from same <code> GameObject </code> - parent class. When other game
 * components need to use these Alien and Player-objects, they use them
 * through the interface declared in <code> GameObject </code> - class.
 * @par
 * Each <code> GameObject </code> is controlled by an instance of <code>
 * BaseController </code>. There are set of AI - classes derived from this class
 * that we use as the "brains" of the Aliens. The <code> Player </code> - object
 * is controlled by an instance of <code> PlayerController </code> - class. It
 * reads the keyboard and mouse statuses and gives the behaving instructions
 * to the <code> Player </code> object.
 *
 * @par About the static classes
 * There are several classes that have only static members (<code>
 * AnimPlayer, BulletStorage, Consts, GameAnims, GfxPool, Map, Settings,
 * SoundPlayer, WarGlobals </code>). Reason why I've used these was that they
 * represents something that must have exactly 1 instance during the gameplay.
 * No more, no less. The <code> GfxPool </code> - class for example. It stores
 * the graphics we've loaded during the game startup. We don't destroy that
 * graphics until we exit the game.
 * @par
 * Reason why we cannot create several instances from <code> Map </code> -
 * class is that I use the default playback library to handle the
 * maps created with MappyWin32. It supports only one map in memory at a time.
 */

#include <allegro.h>
#include "eng2d.h"
#include "game.h"
#include "www_assert.h"
#include <vector>
#include <fstream>
#include <stdexcept>
#include "www_exceptions.h"


/** @fn     int main()
 * @brief   The game entry point.
 * @author  Tomi Lamminsaari
 */
int main()
{
  std::ofstream fout("log.txt");
  fout.close();
  
  LOG_MESSAGE( "Starting Allegro..." );
  allegro_init();
  install_keyboard();
  install_timer();
  install_mouse();
  
  LOG_MESSAGE( "done" );
  
  LOG_MESSAGE( "Setting screen 640,480,16..." );
  if ( eng2d::Display::setUpDB(640,480,16) < 0 ) {
    LOG_MESSAGE( "failed" );
    LOG_MESSAGE( "Setting screen 640,480,15..." );
    if ( eng2d::Display::setUpDB(640,480,15) < 0 ) {
      LOG_MESSAGE( "failed" );
      set_gfx_mode( GFX_SAFE, 320,200, 0,0 );
      alert("Failed to initialize the screen",0,0, "ok",0, 0,0);
      return 0;
    }
  }
  LOG_MESSAGE( "done" );
  

  LOG_MESSAGE( "Init sound device..." );
  eng2d::Sound::install();
  LOG_MESSAGE( "done" );

  WeWantWar::Game* pGame = 0;
  try {
    pGame = new WeWantWar::Game();
    if ( pGame == 0 ) {
      alert("Failed to initialize the game!", "An error log has been created.",
            "log.txt", "OK",0, 0,0);
      return 0;
    }
  
  
    if ( pGame->init() != 0 ) {
      alert("Failed to initialize the game!", "An error log has been created.",
            "log.txt", "OK",0, 0,0);
      delete pGame;
      return 0;
    }

    pGame->run();
    pGame->cleanup();
    delete pGame;
    
  } catch ( std::exception& e ) {
    alert("Exception occured !",
          e.what(),
          typeid(e).name(),
          "OK",0, 0,0);
  } catch ( WeWantWar::Exception& e ) {
    alert("Exception occured !", e.m_message.c_str(), 0, "ok",0, 0,0);
    
  } catch (...) {
    alert("Unspecified exception occured!", 0, 0, "ok",0, 0,0 );
    
  }
  
  return 0;
}
END_OF_MAIN()

